function chris_color_age_pca5
% Shows how pre-sets of cells with similar spiking patterns (from the chris false twins program) project into PCA picture.
%
% Data should be hard-written in the header of the function.

% Nov 30 2014: Revised and revived. 
% Dec 22 2014: Modified from v.5 . Both naive and stimulated cells.
% Dec 23 2014: Now projecting into raw variables as well. Raw values are hard-coded at the very bottom of the file.
%               Trainspace (projection of spike-trains similarity) is also here now.
% Jan 03 2014: Rank-transform of raw values for thresholds. If you need it, chagne if(0) to if(1) below.
% Jan 05 2014: Scores data updated (with varimaxed scores).

list = [1002;1003;1006;2003;4004;6001;13001;13002;13003;14005;17005;19001;19003;23002;23004;24002;2002;3005;
    4001;4003;16005;3001;2004;3002;21002;20001;21001;4002;1004;29006;29001;27001;2001;5001;23001;17003;16004;
    20004;3004;19004;1005;3003;27005;11003;18002;20003;18006;25005;32003;27002;36001;34004;37002;32005;14002;
    29002;77002;18001;7001;31006;25001;18005;34002;36003;16003;33002;37001;29007;11005;9003;5002;23005;30003;
    6003;25004;83003;31005;11006;24001;15001;11001;28003;30002;12002;22003;34001;17001;16002;29003;25006;24006;
    5003;31003;28001;78007;14001;31002;83002;31004;34005;18003;18004;78006;26003;24004;19002;30001;36002;22001;
    77001;22002;33001;9004;17002;11004;34003;32004;23003;35002;14003;24005;27003;16001;82001;24003;35001;78003;
    25003;7003;7002;14004;20002;37003;32001;31001;9001;78002;33003;28002;12004;12001;8001;29004;12003;10001;
    25002;80001;17004;78005;29005;6002;11002;32002;10003;9002;82005;79004;44003;48002;40001;65003;73004;82002;
    79003;41003;42004;79002;73001;64004;72002;82007;82003;67003;79005;72004;43003;79006;42001;45002;65001;82006;
    42003;47002;68001;47001;44002;42002;64003;68002;67004;65002;67005;47003;64001;38001;47004;39001;68003;41001;
    43001;45001;64006;73003;72001;41002;43004;43002;79001;72003;48001;67002;73002;64005;50001;49003;50002;49001;
    49005;49002;49004];

s = [0.0412 -0.2695;0.0607 -1.3641;0.6134 -0.64;0.7801 -0.6858;-0.3069 -0.289;-0.1027 0.8485;0.0333 -1.0067;
    1.0426 0.4466;0.1501 -0.2089;1.1334 0.4029;0.4968 -0.0487;-1.3761 -0.11;0.2061 -0.5888;0.9179 0.0358;
    -0.469 0.1406;-0.0502 0.0452;0.12 -0.4928;0.4438 1.0227;1.1478 -0.6961;0.7908 -0.5899;0.21 0.1308;-0.3965 -0.2085;
    -0.463 0.9678;-0.09 -0.0662;0.0338 0.7148;-1.1383 -1.07;-1.3516 -0.8849;-0.2946 0.0132;0.3632 -0.2613;
    -1.0505 -0.0804;-0.5416 0.0375;-1.7973 -1.6487;-0.2292 -0.1478;0.1654 -0.1302;-0.0956 -0.3899;-0.4739 -1.0852;
    -0.2523 -0.4288;-0.3327 -0.1175;0.7852 0.7114;1.0674 -0.8036;-0.0486 -0.4368;0.4563 -0.006;-1.3639 -0.8835;
    0.4298 -0.1017;-0.4334 -0.2218;0.2689 0.6622;1.3287 0.2033;0.6297 0.4993;0.0637 0.2454;-1.2364 -0.0348;
    -1.5507 0.457;-1.5759 -1.4512;-1.1092 -0.1472;-1.0717 -0.3642;1.2535 0.8487;-1.4497 -1.6374;1.4287 0.0502;
    1.0915 1.8463;1.3103 2.0843;-0.5335 -0.1114;-0.4607 0.0354;1.517 2.3652;-0.3759 -0.0086;-0.9143 -0.3663;
    0.5056 0.1229;-1.6528 -0.4934;-0.2656 -0.9042;-1.2409 -0.9287;1.5416 1.1626;0.9966 0.3378;0.7222 -0.0466;
    -0.7132 -1.0018;-0.9423 0.4335;1.2081 0.6826;-0.5723 2.8617;1.2623 0.3248;-1.1304 -1.2258;1.3484 0.1374;
    -1.4854 -0.8008;0.9785 1.0011;0.6099 0.5223;-0.762 -1.1562;-0.7423 -0.2761;0.2593 -0.2604;-0.6476 2.1865;
    -0.8861 -0.317;0.5338 0.8735;0.8322 0.9172;-1.6072 -1.0731;-0.4711 -1.823;-0.3518 -0.8543;0.659 0.4332;
    -1.0368 -0.2144;-1.8333 -0.5536;1.2143 1.6435;0.9635 0.4152;-0.3522 0.2202;1.738 0.8146;-0.474 -0.9004;
    -0.5205 -0.8699;0.864 0.3541;1.0494 1.3272;1.3352 -0.3391;-1.4141 3.1391;-0.184 -0.0067;-0.2543 0.808;
    -0.6833 0.0581;-0.873 -0.4783;0.2217 0.9579;1.6057 0.8319;-0.6801 -0.7181;-1.0813 1.2572;0.6277 0.3822;
    0.4476 0.6355;0.7271 -0.0355;-1.5875 -0.0648;-0.2241 0.1107;-0.663 -0.0056;-0.5676 0.5579;0.547 1.2492;
    -0.0057 -0.6224;-0.2921 0.4317;0.0908 0.2141;0.423 0.2103;0.0347 -0.9598;-0.2322 0.7794;0.6558 -0.6067;
    0.3549 -0.0776;0.5333 0.0273;0.681 0.2291;0.82 0.26;0.4788 1.1137;-0.8073 -1.7876;-1.161 -0.4886;-0.476 0.9231;
    0.9696 0.0747;0.6831 0.1017;-1.0178 -0.3201;-0.18 -0.6159;0.4007 -0.5946;0.5828 -0.4031;0.0238 1.4899;
    -0.1462 0.2737;0.2862 -0.5715;0.3615 -1.0409;-0.8363 -2.4451;0.5824 0.0609;0.8941 -0.6906;0.9474 -1.1448;
    -0.6839 -0.015;0.6095 0.3337;1.3565 0.1018;-0.1815 0.8227;0.0009 0.5095;-0.0948 -1.7161;1.545 1.9734;
    1.4405 -0.6041;-0.7359 -1.8416;0.8599 0.1186;-1.7897 0.141;1.4755 0.5333;1.7536 -0.3142;1.8229 0.4737;
    1.814 0.2106;-0.4552 -1.9871;-0.4334 -1.6018;1.8572 -0.3323;0.4491 0.8474;2.0347 -0.0453;1.5313 -0.7925;
    1.0466 -0.156;2.0528 -0.1994;2.0069 0.1807;1.4854 0.5927;1.8931 0.229;0.2932 -1.049;1.8764 1.9216;
    -1.2054 -0.0655;-0.2617 -0.4722;1.5774 0.0162;2.1591 -0.6778;-0.6522 -1.2594;0.1194 -0.7696;1.3011 1.2727;
    -1.0784 -0.2096;-0.7215 -1.5259;-0.7698 -0.6332;1.3975 -0.3307;1.5528 0.3266;1.5352 -0.3161;0.7665 0.2487;
    1.2891 0.0936;-1.4317 -0.7376;0.9637 -0.312;0.5875 1.7134;-1.2451 -1.498;-0.9537 0.3891;2.1305 -0.5482;
    -0.9586 -0.1009;-0.1281 0.2003;-1.0084 -0.9521;1.2635 -0.5636;1.5748 -1.0953;1.3917 -0.7202;-0.7476 -0.739;
    -0.1366 -1.1178;0.0141 -0.7943;0.9779 -0.9523;0.5648 -0.2793;0.5453 -1.1129;0.4803 -0.5589;1.18 -1.2497;
    1.1156 -1.1722;0.2087 -0.4846;0.3917 -0.1982;0.8299 -0.3818;0.2804 -1.0139;-0.3604 -1.3402;0.1774 -1.4597;
    0.0138 -0.8555]; % PCA Scores: a mix of actual PCA & ligthbox-projection

mycolor = cbrewer('qual','Set1',4);
 mycolor = mycolor([3 2 4 1],:);            % Reorder in a more meaningful way

group{1} = [25006 34004 44003 22002 29006 1004];    % 1-spike
group{2} = [33001 42001 39001 19002 42003 47001];   % 2-spikes
group{3} = [80001 15001 72003 16002 65002 14003];   % 4-spikes
group{4} = [82005 65001 14001 65003 82002 83002];   % Many-spikes

%%% ------------------------------------ original train space figure

% [traindata,trainid] = trainspace();
% 
% figure;
% hold on;
% g = ~ismember(trainid,[group{:}]);
% plot(traindata(g,1),traindata(g,2),'o','MarkerEdgeColor',[1 1 1]*0.8,'MarkerFaceColor',[1 1 1]*0.9);   
% for(q=1:length(group))
%     g = ismember(trainid,group{q});        
%     plot(traindata(g,1),traindata(g,2),'o','MarkerEdgeColor','none','MarkerFaceColor',mycolor(q,:));    
% end
% legend({'','1','2','3','4'},'Location','NorthEastOutside');
% title('Train similarity');
% hold off;

%%% ------------------------------------- PCA figure
figure;     % PCA space
hold on;
g = ~ismember(list,[group{:}]);
plot(s(g,1),s(g,2),'o','MarkerEdgeColor',[1 1 1]*0.8,'MarkerFaceColor',[1 1 1]*0.9);   
for(q=1:length(group))
    g = ismember(list,group{q});        
    plot(s(g,1),s(g,2),'o','MarkerEdgeColor','none','MarkerFaceColor',mycolor(q,:));    
end
legend({'','1 spike','2-3 spikes','6-7 spikes','>10 spikes'},'Location','NorthEastOutside');
title('Projection to PCA space');
hold off;

%%% ------------------------------------- raw data figure
data = rawdata();
data(:,4) = min(data(:,4),1300);  % Bringing a rogue outlier into the box (for iNa_pA)
data(:,2) = min(data(:,2),4.3);

% 1     2       3       4       5       6       7       8       9       10
% id	rm      cm      nath	napa	ksth	kspa	ktth	ktpa    sampa

mylabels = {'','Rm, MOhm','Cm, pF','iNa threshold, mV','iNa, pA','iKs threshold, mV','iKs, pA','iKt th, mV','iKt, pA','Spike-amplitude accomodation'};
% xi = 4; yi = 7;   % Both currents
xi = 4; yi = 6;   % Both thresholds (19% together)
% xi = 5; yi = 7;   % Both currents (8% together)
% xi = 10; yi = 6;  % sampa and ksth - BAD
% xi = 2; yi = 5; 	% rm and napa (3d and 4th in terms of explaining stuff; 10% together)
% xi = 2; yi = 3;   % rm and cm (15% together)

nx = data(:,xi);    % Just in case we decide to transform them
ny = data(:,yi);

if(1)   % Rank-transform
    oldx = nx;              
    oldy = ny;
    [~,i] = sort(nx);	[~,newx] = sort(i);     p = polyfit(newx,oldx,3);	nx = polyval(p,newx);
    [~,i] = sort(ny);	[~,newy] = sort(i);     p = polyfit(newy,oldy,3);	ny = polyval(p,newy);
end

figure;     % Raw variables
hold on;
g = ~ismember(data(:,1),[group{:}]);    % Fins cells based on their ids
plot(nx(g),ny(g),'o','MarkerEdgeColor',[1 1 1]*0.8,'MarkerFaceColor',[1 1 1]*0.9);  
for(q=1:length(group))
    g = ismember(data(:,1),group{q});        
    plot(nx(g),ny(g),'o','MarkerEdgeColor','none','MarkerFaceColor',mycolor(q,:));   
    % set(gca,'XScale','log','YScale','log');
end
legend({'','1 spike','2-3 spikes','6-7 spikes','>10 spikes'},'Location','NorthEastOutside');
title('Projection to raw variables');
xlabel(mylabels{xi});
ylabel(mylabels{yi});
hold off;

end



function data = rawdata()

% 1     2       3       4       5       6       7       8       9       10      11
% id	rm      cm      nath	napa	ksth	kspa	ktth	ktpa    sampa   samp


data = [1002	0.72	12.4	-27.3	267.7	-28	377.4	7.7	216.7	2.315421348	23.03532663
1003	1.2	7.5	-19.6	88.9	-29.8	234.1	6.7	44.8	2.315421348	23.03532663
1006	1.2	14	-5.9	243	-25	434.9	2.2	47.9	2.315421348	23.03532663
2003	2	10	-16.6	252.7	-11.1	580.1	9.7	172	2.315421348	23.03532663
4004	0.9	17.9	-17.8	274.5	-19	415	-16.9	108.4	2.315421348	23.03532663
6001	1.1	19.9	-22.9	612.8	-28	711.9	-13.9	184	2.315421348	23.03532663
13001	1.1	12.4	-20.1	251.2	-9.9	196.7	-9.9	54.7	2.315421348	23.03532663
13002	0.845	18	-11.5	410.8	-26.3	428.2	-7.4	300.3	2.315421348	23.03532663
13003	0.928	17	-17.6	314.6	-17.8	386.6	-12.8	150.8	2.315421348	23.03532663
14005	0.932	18	-7.5	512.5	-18.4	346.9	-11.1	258.4	2.315421348	23.03532663
17005	1.6	13.8	-16.7	367.5	-18	424.7	-16.7	202.3	2.315421348	23.03532663
19001	1.3	15.9	-38	476.8	-20.4	562.5	-16.2	58.9	2.315421348	23.03532663
19003	1.3	11.6	-18.1	407.1	-8.4	263.7	-13.8	141.1	2.315421348	23.03532663
23002	1	16	-9.3	309.1	-17.7	398.5	-19.4	303.5	2.315421348	23.03532663
23004	0.737	21	-25.1	435.1	-22.3	397.3	-17.5	89.4	2.315421348	23.03532663
24002	0.843	15.6	-20	571.8	-11.3	296.9	-18.8	141.3	2.315421348	23.03532663
2002	2.1	14.3	-17.6	226.7	-16.3	260.7	-22.9	63.4	3.412	12.36
3005	1.2	19.7	-27.4	585.5	-19.4	665.5	-0.1	146.5	2.112	38.749
4001	1.7	19	-7.5	317.2	-15.3	365.3	-16.7	243.5	1.146	12.512
4003	1.6	14	-17.4	188.2	-10	243.8	-19.4	141.4	1.497	15.895
16005	1.6	12	-28.4	325.6	-21.7	478.9	-29	116.2	1.744	20.242
3001	1.3	19	-23.2	350.4	-27.6	438.5	11.5	87.5	3.747	14.339
2004	0.935	14.2	-19.7196347	411.5438356	-18.78995434	539.2009132	-15.36757991	166.5127854	2.504	23.834
3002	1.1	17.4	-28.7	221.1	-30.5	280.8	-2.8	46.5	3.998	20.203
21002	0.654	18	-18.7	656.3	-17.6	470.2	-14.8	235.7	2.225	18.489
20001	1.1	14.6	-25.8	324	-9	310	-15.7	108.9	5	15.747
21001	1.5	13	-19.5	425.1	-10.8	431.6	-22.5	81.6	7.561	17.103
4002	1.2	20	-16.2	354.5	-17.3	466.7	-24	83.5	2.156	14.404
1004	0.592	16	-6.1	180.9	-19.4	313.3	-19.2	145.2	2.315421348	19.523
29006	0.45	14	-28.8	379.6	-19.3	523.7	-26.9	64.6	2.315421348	10.295
29001	0.8	17	-27.6	439.6	-18.9	608.3	-28.1	106.9	1.934	23.641
27001	1.2	10.9	-15.5	95.2	-16	444.7	-24.1	101.4	8.947	9.532
2001	1.3	19	-18.6	281.9	-19.9	396.3	-22.4	124.1	1.32	8.353
5001	1.24	13	-27.4	314.4	-28.9	432.4	9.9	39.9	1.709	14.954
23001	1.7	13.4	-19.4	549.3	-17.6	502.9	-19.7	226.1	1.529	22.339
17003	3.3	9.1	-18.4	261.2	-16.2	562.1	-23.9	136.6	2.315421348	23.03532663
16004	0.922	14	-27.5	271	-20	534.8	19	64.8	3.801	18.241
20004	1.1	13	-19.2	447.3	-17.6	322.7	-20.7	163.3	2.315421348	11.985
3004	1	19	-16.5	394.9	-18.7	680.9	-11.6	254.1	1.463	21.21
19004	1.8	10.36	-17.8	183.6	-6.9	240.1	-13.3	344.1	2.315421348	23.03532663
1005	1.5	11.3	-18.1	296.2	-29.8	295.9	-20.2	141.8	2.315421348	5
3003	1.5	17	-18	418.6	-20.5	401.7	-9.6	154.4	1.58	18.703
27005	1.3	12	-27.5	180.6	-21	657.8	-30.3	50.6	2.599	9.969
11003	1.2	15	-17.8	322.1	-18	297.3	-17.8	303.4	2.315421348	23.03532663
18002	1.3	15.4	-17.6	538.8	-19.5	634.2	-15.5	170.1	2.315421348	23.03532663
20003	0.964	17	-16.1	410.5	-22.6	478.4	-18.6	177.6	2.315421348	23.03532663
18006	1.3	13.6	-15.3	377.1	-9.3	567.8	-12.4	552.6	0.82	15.128
25005	1.3	12.81	-12.1	705.1	-16.1	1022.7	-0.9	191.5	2.725	7.019
32003	1.4	15.8	-18.5	381.7	-16.6	411	-19.7	232.1	1.642	15.371
27002	1.3	15.5	-17.3	312.8	-28.8	498.5	-17.3	170	2.117	23.946
36001	0.875	17.4	-28.9	601.6	-19.2	448.7	-24.8	46.7	3.383	18.595
34004	1.1	11	-20.6	354.9	-20.4	401.7	-23.7	154.6	2.315421348	12.777
37002	1.5	17.3	-27	358.2	-28	622	-27.2	176.8	1.609	7.497
32005	1.5	13	-27	364.4	-16.9	620.5	-28.9	84.9	2.315421348	7.182
14002	1.2	15	-17.1	533	-18.7	530.5	-16.3	277.2	1.962	33.19
29002	1.4	11	-28.9	122.8	-18.7	195.1	-58.8	11.4	2.315421348	16.083
77002	1.8	5.48	-29.1	288.3	-10.4	864.2	21.6	139.4	2.249	36.926
18001	0.6	19	-20	914.2	-15.5	601.1	-17.9	280.3	1.22	28.782
7001	1.1	17	-16	607.3	-25.6	1010.5	3.1	248	1.743	12.338
31006	1.1	13.5	-29	610.4	-16.8	646.3	-27.4	209	2.484	23.499
25001	1.2	15.2	-17.3	363.2	-21.7	1025.5	6.8	265.7	2.315421348	21.327
18005	0.822	20	-14.2	1006.9	-7.1	893.8	-16.5	476.9	1.322	32.113
34002	0.783	17.4	-28.4	441.8	-27.6	380.5	-27.4	147.9	2.294	16.632
36003	0.522	17.6	-29.6	364.6	-25.5	365	-28.7	110.3	1.594	16.774
16003	0.841	17	-18.7	441.9	-26.9	427.7	-21.3	110.5	1.165	19.392
33002	0.762	14	-28	262	-19	357.3	-27.9	132	2.315421348	11.007
37001	1.5	14.1	-23.3	226	-18.3	547.4	-24.8	317.7	0.851	8.718
29007	1.1	13	-22.6	221.8	-18.4	722.4	-14	209	5.648	9.42
11005	1.3	14	-16.5	727.3	-9.3	734.7	-11.8	456.3	1.447	20.351
9003	1.1	14.9	-17.4	337	-16.1	354.7	-19.7	167.4	1.241	27.204
5002	0.95	11.5	-36.6	360.8	-37.1	248.5	3.2	270.2	3.109	17.029
23005	3.4	10	-24.5	285.3	-15.5	666	-25.2	200.6	2.315421348	7.734
30003	0.72	13.5	-28.9	427.6	-20.6	402.9	-27.7	162.6	2.641	14.313
6003	1.2	19	-15.4	631.4	-19.9	513.4	-15.1	395.7	1.036	38.156
25004	0.62	24.11	-17.2	1283.3	-28.4	1221.8	-3.3	124.7	1.644	24.353
83003	0.88	7.74	-13.2	370.7	-17.7	850.3	-11.5	233	2.045	40.7
31005	1	15	-17.3	232.1	-17.7	325	-19.6	171.4	1.831	9.104
11006	1	17.5	-11.3	366.6	-21.5	307.7	-9.5	374.3	1.859	13.929
24001	2.6	14	-28.8	327.8	-18.9	317.2	-26.6	86.3	2.315421348	16.523
15001	0.802	16	-17.3	657.3	-13.2	470.7	-16.1	429.6	1.319	33.238
11001	1.2	17.9	-16.7	280.8	-25.1	571.5	-19.4	211.1	1.733	12.626
28003	1.1	14	-20.9	201	-17	485	-21.1	83.8	2.912	9.878
30002	1.1	15.7	-29.5	466.1	-18.9	435.7	-28.3	70.8	2.122	27.924
12002	1.2	14.7	-17.1	273.3	-20.4	347.9	-17.3	105.7	1.882	13.062
22003	0.5	21.2	-19.7	1016.3	-18.9	611.4	-18.9	354.8	1.151	17.809
34001	1.1	15	-27.8	396.1	-19.4	612.5	-30.1	136.9	3.381	19.48
17001	1.2	13	-27	716.7	-13.5	685.8	-19.3	192.2	1.545	38.731
16002	1.5	21	-16.3	507.1	-20.1	523.4	-17.8	270.8	1.953	30.212
29003	1.1	13	-18.7	255.8	-19.8	593.7	-27.5	65.5	5.35	10.62
25006	3.1	9.75	-12.3	300.9	-8.1	347.6	2.1	96.9	2.315421348	13.236
24006	1.3	12.93	-18.2	463.1	-15.6	374.8	-8.2	199.1	3.818	14.165
5003	0.685	19.3	-17.8	448.6	-22.1	436.7	-21.6	149.9	1.377	16.985
31003	1	13.3	-27	381.6	-16.4	663.4	-26.2	260.4	2.082	15.574
28001	1.3	16	-18	206.1	-18	763.6	-29.1	81.8	6.312	4.13
78007	0.3527	10.38	-15.6	953.1	-25.7	1242.1	-24.5	57.8	1.394	60.465
14001	0.915	20	-19.1	401.8	-19.6	397.7	-21.3	117.7	1.386	16.087
31002	1.1	14.4	-19.2	496.9	-16.7	481	-20.1	138.6	1.657	22.329
83002	0.7455	8.15	-15.7	431.1	-19	975.2	-18.2	118.4	1.47	40.029
31004	1.2	11	-17.9	392.9	-17.2	495.8	-18.5	231.3	1.644	10.203
34005	0.66	13.6	-13.5	153.2	-35.9	486.3	-20.1	257.3	1.069	7.406
18003	2.3	15	-19	640.6	-15.5	512.9	-18.3	222.3	1.128	31.132
18004	1.6	16.7	-25.9	830.3	-16.6	606.3	-15.3	220.3	1.301	29.079
78006	1.8	7.76	-15.9	201.7	-7.2	561.2	-19.7	117.8	1.462	27.649
26003	0.504	24.8	-18.7	1650.4	-17.7	1600.7	-19.8	446.5	2.315421348	11.973
24004	1.2	13.7	-18.9	641.5	-8.5	285.6	-13.1	253.1	1.484	24.296
19002	0.82	15.6	-27.6	635.9	-12.2	504.2	-18.4	148.8	2.017	11.126
30001	1	22	-27.5	404.6	-20.5	594	-27.6	183.4	2.678	22.868
36002	1.2	17.5	-27.5	420.2	-20.5	608.7	-29.5	87.3	4.764	21.088
22001	0.83	19.25	-30	676.2	-11.6	357	-15.2	126.2	1.827	19.579
77001	0.7699	7.14	-16.8	443.9	-16.3	926.8	2.6	173.1	1.756	48.991
22002	0.884	19	-9	312.7	-26	279.6	-14.4	207.5	2.315421348	11.047
33001	0.875	19.5	-28.3	651.5	-35.2	858.8	-27.7	43.1	4.348	20.162
9004	1.24	14	-17	322.7	-17.6	580.3	-11.1	155.3	1.877	18.114
17002	0.935	17	-18.8	584.1	-18.9	579.9	-20.7	121	2.005	23.725
11004	1.4	16	-17.4	223.6	-19	231.3	-20	141.3	1.744	30.827
34003	1.3	15	-19.5	267.5	-35.8	360.7	-22.8	105.4	2.315421348	2
32004	0.631	16	-29.6	500.7	-25.4	444.6	-28.3	173.7	1.24	17.253
23003	1.8	14.5	-19	768.8	-15.9	556.9	-20.9	243.4	3.317	21.79
35002	0.7	17	-28.1	756.1	-26.7	1268.3	-30.4	120.4	1.599	25.014
14003	0.48	17	-18.8	730.3	-11	492.2	-21.6	143	1.306	38.064
24005	1.5	13.11	-17.7	227.2	-26.4	299.1	-16.3	201.9	2.848	7.656
27003	0.5	19.16	-19.9	443.3	-22.2	373.2	-18.3	102.3	1.765	10.295
16001	1.3	16	-27.2	362.9	-21	709.1	-27.7	120.8	1.968	19.091
82001	0.8105	12.41	-27.8	578	-10.3	457.3	-19.9	115.4	1.65	41.494
24003	1.1	12.8	-19	308.3	-19.9	215.2	-17.2	128	2.19	15.625
35001	0.869	17.5	-28.9	595.1	-36.3	665.9	-28.4	58.9	2.737	21.332
78003	0.9693	9.97	-17.8	182.1	-37	185.9	-13	92.5	1.875	23.977
25003	1.7	13.7	-11.9	526.5	-15	1055	-18.2	112.2	2.329	4.565
7003	1.5	14.8	-24.6	235.8	-17.1	572.6	-1.2	352.4	1.35	14.975
7002	1.4	19.5	-26.5	321.2	-26.6	321	12.6	67.4	1.877	34.969
14004	1.9	15	-18.9	620.6	-17.3	418.9	-5.6	169.7	1.617	37.955
20002	1.1	17.8	-18.3	624.8	-20.2	471.4	-19.1	162.2	1.897	12.983
37003	3.3	5.8	-12.7	381.7	-16	675.1	9.6	85.5	2.315421348	7.619
32001	1.1	15.5	-25.6	441.7	-18	991.6	-20.4	274	4.048	14.781
31001	0.83	15	-28.7	706.9	-15.8	1041.5	-24.2	81.7	3.696	20.671
9001	1.9	19.65	-12.7	210	-8.2	308.1	-17.1	414	1.136	30.709
78002	0.437	7.87	-16	478.9	-22.1	503.9	-15.1	102.3	2.295	46.183
33003	1.1	16.5	-24.7	451.4	-21.7	667.1	-24.7	57.4	4.966	8.779
28002	1	7.4	-18.4	362.1	-20.1	728.5	-11.4	108.1	2.689	14.73
12004	1.5	11	-16.8	323.2	-8.6	370.6	-1	101.1	2.016	13.559
12001	1.2	14	-17.6	129.7	-21	149.5	-14.6	181.1	2.381	8.48
8001	0.409	18.8	-27.6	784.3	-22	1078.8	-0.8	210	2.329	22.98
29004	1.1	16	-22.7	605.1	-17.9	1545.7	7.8	169.5	2.387	18.3
12003	1.2	11.5	-16.7	326.3	-8.4	326.1	-12.6	243.1	2.7	16.972
10001	2.1	11.5	-16.8	168.8	-16.2	506.9	-23.2	144.8	2.433	26.127
25002	4.2	6.6	-16.1	166.9	-7.6	240.8	0.9	81.6	2.315421348	18.141
80001	0.3641	9.86	-17.7	590.5	-17.3	819.5	-15.1	138.8	1.842	44.393
17004	1.6	16.5	-6.8	227.2	-20.1	419.7	-17.5	313.1	1.105	20.508
78005	0.7563	7.03	-19.6	249.4	-21.1	217.7	-17.8	95.5	0.832	19.175
29005	1.2	14.3	-25.1	731.9	-16.7	1047	2.9	223.7	3.003	25.533
6002	1.8	17.5	-10.2	549	-8.9	836.7	-12.1	120.8	0.987	22.021
11002	1.2	16	-8.9	258.2	-15.8	437.8	-13.4	291.8	1.196	16.536
32002	1	16.45	-29	580.7	-17.1	663.5	-25.4	155.9	2.057	22.614
10003	2.2	15.8	-18.9	842.9	-16.7	667.6	-24.4	108.1	3.545	32.497
9002	2.9	10.9	-18.5	175.3	-7.5	234.7	17.4	87.8	3.706	14.417
82005	0.5103	15.3	-18	859.4	-15.4	745.5	-15.4	237.8	1.191	58.044
79004	1.3	8	-17.4	277.2	-8.2	529.9	-19.4	81.4	1.72	43.111
44003	1.8	8.23	-25.1	224.6	-15.6	570.9	-3.5	78.4	2.315421348	18.005
48002	1.8	12.46	-26.7	227.4	-21.9	413.9	-24.3	95.2	1.997	27.008
40001	1	15.9	-18.8	468.3	-28.9	531.2	-18.8	101.1	3.036	13.031
65003	0.448	12.57	-14.3	460.6	-23	475.1	-17.9	165.4	1.475	39.754
73004	0.8008	8.3	-11.1	332.8	-6.6	662.5	-4.1	162.1	3.607	44.881
82002	0.4696	11.37	-15.7	545.9	-8.2	710.9	-12.4	215.4	1.405	53.365
79003	0.3886	12.75	-20	475.5	-9.5	362.2	-7	127.7	1.63	44.881
41003	8.1	5.7	-30	371.7	-24	339.8	-28.8	75.5	2.315421348	12.828
42004	2	15.56	-7.9	186.5	-12.4	441.1	-17.4	90.1	3.672	9.013
79002	0.6746	9	-7.4	278.3	-10.2	429.9	-14.7	195.4	1.758	40.141
73001	0.954	7.7	-16.7	420.8	-20	1058.5	8.3	141.7	1.778	41.951
64004	0.695	9.72	-9.7	273.7	-9.2	498.5	-12.1	153.4	2.437	44.779
72002	1.2	8.96	-4.1	135.9	-6.9	856.5	-16.5	172.2	2.304	36.428
82007	1.1	11.57	-17.1	296.4	-8.4	742.8	-20.3	149.3	2.686	35.38
82003	0.7637	12.77	-7.4	277.5	-20.6	374.8	-13.3	353.4	0.484	39.012
67003	0.5306	9.12	-20.4	415.1	-16.8	403.6	-6.4	116.6	1.244	52.317
79005	0.5029	10.19	-16.3	416.9	-9.6	956	-21.2	113.1	1.316	48.859
72004	1.3	11.11	-14.7	641.1	-7.5	717.9	-12.5	162.1	2.065	43.111
43003	1.1	8.65	-30.7	143.3	-19.1	266.8	0.9	238.1	2.425	10.966
79006	0.7168	11.46	-14.4	1042.3	-8.6	1213.3	-15.7	99.3	1.442	44.25
42001	2	17.11	-27.9	350.3	-21.9	768.7	-27.7	87.7	3.735	12.746
45002	0.6674	8.26	-9.3	263.1	-20.6	334.7	-12.1	154.8	1.747	8.006
65001	0.4524	13.47	-9.1	357.7	-19.3	333.8	-13.7	136.2	1.269	44.362
82006	0.7014	12.07	-7.7	285	-31.7	435.2	-11.9	365	1.05	36.875
42003	3.3	14.37	-27.7	331.5	-19.5	554.8	-11	213.7	3.617	12.451
47002	2.3	11.97	-16.5	147.1	-38.9	208	-11.3	272.8	2.315421348	6.215
68001	0.5366	9.91	-15.5	468.7	-15.3	1428.6	-9.3	222.9	1.373	32.644
47001	1.8	12.69	-26.2	417.9	-25	531.4	-23	105.3	3.921	16.571
44002	0.972	10	-9.6	183.4	-11.7	321.6	-13.9	169.7	2.315421348	6.5
42002	2.3	2.6	-26.5	633	-36.6	678.3	-9.2	208	0.81	15.798
64003	0.7417	8.63	-17.4	274.1	-9.3	468.8	-21.7	74.7	2.289	38.31
68002	0.9567	8.23	-15.7	295.2	-8.6	1194.6	-21.4	71.1	1.597	42.582
67004	0.5919	5.5	-17.5	439.9	-16.2	620.8	-21.4	36.8	1.51	42.135
65002	0.3733	12.73	-15.5	539.3	-10.2	635.8	-18.7	182.4	1.781	46.082
67005	0.6283	6.77	-18.5	392.6	-15.7	748.2	-18.1	103.4	1.863	43.681
47003	2.7	16.13	-27.5	181.8	-34	313.7	-9.9	159.4	1.924	3.713
64001	1.5	7.63	-17.6	236.3	-6.6	881.3	-20.7	104	4.823	42.918
38001	1.3	21.7	-29.4	808.1	-28.9	824.7	-28.3	94	1.403	34.861
47004	0.9881	11.24	-26.8	316.1	-27.5	266.9	-5.4	301.9	2.844	13.662
39001	1.5	21.23	-27.3	482.4	-30	795.5	-20.4	57.6	2.788	10.488
68003	1.1	6.99	-6	225.3	-18.1	512.9	-15	180.3	1.264	39.368
41001	3.2	27.5	-29.5	642.3	-35.7	547	-7.7	143.4	4.442	24.017
43001	1	15	-14	526.2	-17.3	778.7	-11.6	228.2	2.459	10.722
45001	0.9888	9.46	-29.4	518.9	-26	536.9	-21.7	61.2	4.553	18.768
64006	0.6783	9.03	-13.4	302.9	-35.7	664.8	-12.1	501.5	3.621	22.542
73003	1.8	5.41	-17.7	286	-18.7	263.2	-14.1	92.9	1.625	34.597
72001	0.4728	8.18	-9.7	249.5	-27.4	241.5	-3	182.8	3.424	43.193
41002	2.7	30.8	-21.5	449	-25.6	377.1	-18.3	202.6	1.843	32.41
43004	1.3	9.78	-24.4	282.7	-28.3	476.8	-15.6	348.3	2.809	10.885
43002	0.41	13.4	-7.3	274.3	-25	242.1	-8.4	62.2	1.642	16.347
79001	0.4797	9.76	-18.1	272.7	-9.7	235.3	-9.8	72.2	1.889	40.619
72003	0.565	11.41	-16.4	320.7	-18.7	439.8	-23.6	207.7	1.445	37.221
48001	0.8789	10.32	-26.7	207.2	-20.7	423.2	-25.8	37.3	1.663	23.326
67002	0.3423	8.08	-20.3	183.7	-17	700.6	23.8	52.6	1.419	24.75
73002	1.2	5.92	-18.8	249.1	-17.9	223.5	-17.9	87	4.01	43.233
64005	0.6616	9.68	-8.5	172.9	-17.1	350.2	-11.6	112.6	7.115	27.395
50001	1.8	11.47	-27.8	286.6	-16.2	511.8	-25.7	212.6	2.099	17.293
49003	1.1	14.54	-28.5	513.7	-10.3	392.2	-15	138.5	1.856	37.435
50002	0.922	14.23	-18.2	292.1	-8.7	440.4	-18.9	198	1.552	28.229
49001	1	14.2	-19.5	223	-17.2	314.7	-22.6	118.3	1.557	25.533
49005	1	15.3	-18.6	257.7	-18.1	395.8	-23	118.9	5.459	12.583
49002	2.7	13.29	-24.3	164.9	-10.3	344.9	-21.9	152.3	1.736	20.854
49004	1.6	15.56	-23.7	274.7	-18.5	292.5	-17.5	186.9	2.12	19.653];

end


function [data,id] = trainspace()

data = [-14.7217 -4.6785;-22.6661 -3.9319;-13.5176 3.0287;-15.8753 0.87;-14.7151 -4.7648;-19.2438 -2.1003;
    -22.5604 1.5;-20.7648 1.555;-16.8546 -6.1277;-19.8363 0.7664;-6.1772 -6.4429;-13.7529 -4.2772;-12.0059 -2.8036;
    -28.0873 -0.8921;-19.1469 -1.0521;-31.6617 2.423;-2.1913 -3.5538;-21.1374 -0.5819;-18.4998 2.3424;-35.6418 3.7179;
    -30.623 3.4726;-14.5236 2.5231;-2.7835 6.0035;2.736 -2.6745;-26.0847 -0.7094;-5.597 -8.9042;-19.8138 1.1555;
    -23.1155 0.0066;-29.5925 3.423;-21.2152 -0.6122;-3.4691 -6.4205;-20.1488 -1.0207;-18.5768 2.8735;-4.7435 2.2323;
    9.3 9.0387;24.563 -10.5397;-22.0838 -0.1526;-19.3378 -2.525;-15.8471 -5.7297;-35.0272 0.4239;-20.1761 -1.2387;
    -25.5628 4.4151;-22.1421 -1.1561;-16.673 -4.1614;-33.6446 2.668;-29.6389 3.157;-25.2751 -2.9954;-22.379 3.2541;
    -17.9804 -0.0679;-34.6331 3.6338;-26.7293 1.8723;-22.0861 -0.8557;13.5211 -14.8356;21.9363 -17.0168;
    -10.2297 -6.1364;11.3399 -2.248;0.1066 -12.0446;-10.0932 -4.893;-15.9321 1.1368;18.7746 -5.9938;
    23.2194 -7.0228;42.6956 -14.4119;32.4051 -8.0086;65.4658 -1.6803;33.9417 -3.0472;12.9343 9.6442;
    47.6173 14.2054;20.8191 2.0961;44.1799 10.0869;-16.5922 2.1537;62.2599 -4.7746;33.2445 -21.7498;
    41.2655 -21.8223;56.7188 8.2904;64.0962 -5.7605;59.9446 -10.7135;8.8753 -11.3962;37.7892 -8.9389;
    9.5231 5.1562;68.9037 -0.8266;19.8672 -7.9291;17.9484 -14.9878;19.3661 -8.2938;49.0174 -11.8023;
    31.8805 -2.2858;31.4238 -11.898;-0.7305 1.8846;13.3997 -10.2589;-14.605 -4.3609;49.0684 -8.4661;
    29.7201 19.3864;28.0498 -8.1586;39.5797 -2.5912;72.7504 1.3634;24.8438 -21.0415;51.0655 0.392;
    75.5128 43.2938;11.5118 4.1502;15.0234 -16.6101;59.4001 2.2343;34.8655 10.4383;51.1625 16.8692;
    23.4852 8.4346;19.6152 -0.1786;56.6608 6.7926;37.145 -7.0311;-32.6201 3.489;-28.907 0.6717;-17.8886 -10.6183;
    -26.5779 3.0047;-27.9758 -0.2023;-18.7171 -4.6172;0.4142 -6.0191;-20.1241 -0.732;-12.8975 -6.5636;
    -32.6644 2.1167;-18.7603 1.806;-6.3117 -2.1191;-8.4194 -0.422;-29.6524 3.1805;-26.7433 2.0201;
    -7.2365 9.8826;-29.6039 3.039;-35.6418 3.7179;-28.5686 0.7549;-12.43 -1.8315;-15.1862 -1.4583;
    -24.2361 -1.4079;-28.3824 3.5144;0.2659 -0.6288;4.008 0.5291;19.3536 2.4875;-0.3593 4.8021;-16.8283 0.3027;
    3.1343 -6.0699;-2.8134 -7.1708;-2.7193 -8.9412;8.1455 12.5559;35.7159 9.9928;17.3328 19.6703;1.1372 5.5117;
    0.0164 -4.7789;14.816 -1.8038;-12.3713 5.0596;13.8391 -4.3641;-7.1904 2.5898;36.8745 15.5015;-5.0157 7.3747;
    -2.9339 -5.0341;-0.9953 -6.4971;-2.842 8.6839;4.9952 4.6366;10.9662 -4.4537;42.1674 4.912;16.788 14.328;
    -16.5281 3.2959;-7.4846 2.8847;-10.3916 -1.5483;-12.0732 1.7909;44.881 20.7028;32.6053 -0.0436;8.6981 5.797;
    16.3284 -11.4224;17.4113 8.2271;-9.3564 2.5582;12.7301 3.8346;-5.0944 2.7116;-20.9754 -0.6989;-25.4103 3.0237;
    17.4028 -12.8432;0.8288 -0.8222;0.9463 0.0045;26.5735 30.395;30.4177 -11.353;17.1121 4.2436;17.0706 7.84;
    -10.6138 0.478;-19.6589 1.955;-26.1594 -0.0782;-19.4931 3.077;-27.7557 1.8891;-26.0975 -0.461;-18.7136 2.5273;
    -4.4464 0.8235;-30.621 3.6857;-14.1594 4.6579;-16.2519 -2.6304;-23.0812 -0.3316;-27.7186 2.2427;-27.7437 2.3373;
    -24.6558 1.949;-14.2759 -0.6012;-25.5833 3.3212;-16.6402 2.7958;-32.6165 3.4785;-32.6261 3.5582;-8.0726 -0.3911;
    -30.5651 3.4236;1.3641 6.0086;-29.6588 3.1251];

id = [30001;30002;30003;31001;31002;31003;31004;31005;31006;32001;32002;32003;32004;32005;33001;33002;33003;34001;
    34002;34003;34004;34005;35001;35002;36001;36002;36003;37001;37002;37003;38001;39001;40001;43001;43002;43003;
    43004;41001;41002;41003;42001;42002;42003;42004;44002;44003;45001;45002;47001;47002;47003;47004;48001;48002;
    49001;49002;49003;49004;49005;50001;50002;64001;64003;64004;64005;64006;65001;65002;65003;67002;67003;67004;
    67005;68001;68002;68003;72001;72002;72003;72004;73001;73002;73003;73004;77001;77002;78002;78003;78005;78006;
    78007;79001;79002;79003;79004;79005;79006;80001;82001;82002;82003;82005;82006;82007;83002;83003;26003;27001;
    27002;27003;27005;28001;28002;28003;29001;29002;29003;29004;29005;29006;29007;1003;1004;1005;2001;2002;2004;
    3001;3002;3003;3004;3005;4001;4002;4003;5001;5002;5003;6002;6003;7001;7002;7003;8001;9001;9002;9003;9004;10001;
    10003;11001;11002;11004;11005;11006;12001;12002;12003;12004;14001;14002;14003;14004;15001;16001;16002;16003;
    16004;16005;17001;17002;17004;18001;18003;18004;18005;18006;19002;20001;20002;20004;21001;21002;22001;22002;
    22003;23001;23003;23005;24001;24003;24004;24005;24006;25001;25002;25003;25004;25005;25006];

end